<?php
declare(strict_types=1);

namespace App\Controller;

use App\Model\Table\UsersTable;
use Cake\I18n\DateTime;
use Cake\I18n\FrozenTime;
use Cake\Mailer\Mailer;
use Cake\Utility\Security;
use function PHPUnit\Framework\equalTo;

/**
 * Auth Controller
 *
 * @property \Authentication\Controller\Component\AuthenticationComponent $Authentication
 */
class AuthController extends AppController
{
    /**
     * @var \App\Model\Table\UsersTable $Users
     */
    private UsersTable $Users;

    /**
     * Controller initialize override
     *
     * @return void
     */
    public function initialize(): void
    {
        parent::initialize();

        // By default, CakePHP will (sensibly) default to preventing users from accessing any actions on a controller.
        // These actions, however, are typically required for users who have not yet logged in.
        $this->Authentication->allowUnauthenticated(['login', 'register', 'forgetPassword', 'resetPassword']);

        // CakePHP loads the model with the same name as the controller by default.
        // Since we don't have an Auth model, we'll need to load "Users" model when starting the controller manually.
        $this->Users = $this->fetchTable('Users');
    }

    /**
     * Register method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */


    /**
     * Forget Password method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful email send, renders view otherwise.
     */

    public function forgetPassword() {
        if ($this->request->is('post')) {
            // Find user by email
            $user = $this->Users->findByEmail($this->request->getData('email'))->first();

            if ($user) {
                // Generate nonce and set expiry (7 days)
                $user->nonce = Security::randomString(128);
                $user->nonce_expiry = FrozenTime::now()->addDays(7);

                if ($this->Users->save($user)) {
                    // Setup the mailer
                    $mailer = new Mailer('default');
                    $mailer->setEmailFormat('both')
                        ->setTo($user->email)
                        ->setSubject('Reset Your Enchanted Fabric Account Password')
                        ->setViewVars([ // Set view variables directly on the Mailer instance
                            'nonce' => $user->nonce,
                            'email' => $user->email,
                        ])
                        ->viewBuilder()
                        ->setTemplate('reset_password');  // Use your email template here

                    // Remove this debug line
                    // debug($mailer->getFrom());

                    // Send the email
                    try {
                        if ($mailer->deliver()) {
                            $this->Flash->success('Please check your email for the password reset link.');
                        } else {
                            $this->Flash->error('Error sending password reset email. Please try again.');
                        }
                    } catch (\Exception $e) {
                        $this->Flash->error('Email error: ' . $e->getMessage());
                        \Cake\Log\Log::error('Email error: ' . $e->getMessage());
                    }
                } else {
                    $this->Flash->error('Failed to generate reset link. Please try again.');
                }
            } else {
                // Always show this message to avoid information leakage
                $this->Flash->success('If an account with that email exists, you will receive a password reset email.');
            }

            return $this->redirect(['action' => 'login']);
        }
    }

    /**
     * Reset Password method
     *
     * @param string|null $nonce Reset password nonce
     * @return \Cake\Http\Response|null|void Redirects on successful password reset, renders view otherwise.
     */
    public function resetPassword(?string $nonce = null)
    {
        $user = $this->Users->findByNonce($nonce)->first();

        // If nonce cannot find the user, or nonce is expired, prompt for re-reset password
        if (!$user || $user->nonce_expiry < DateTime::now()) {
            $this->Flash->error('Your link is invalid or expired. Please try again.');

            return $this->redirect(['action' => 'forgetPassword']);
        }

        if ($this->request->is(['patch', 'post', 'put'])) {
            // Used a different validation set in Model/Table file to ensure both fields are filled
            $user = $this->Users->patchEntity($user, $this->request->getData(), ['validate' => 'resetPassword']);

            // Also clear the nonce-related fields on successful password resets.
            // This ensures that the reset link can't be used a second time.
            $user->nonce = null;
            $user->nonce_expiry = null;

            if ($this->Users->save($user)) {
                $this->Flash->success('Your password has been successfully reset. Please login with new password. ');

                return $this->redirect(['action' => 'login']);
            }
            $this->Flash->error('The password cannot be reset. Please try again.');
        }

        $this->set(compact('user'));
    }

    /**
     * Change Password method
     *
     * @param string|null $id User id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
//    public function changePassword(?string $id = null)
//    {
//        $user = $this->Users->get($id, ['contain' => []]);
//        if ($this->request->is(['patch', 'post', 'put'])) {
//            // Used a different validation set in Model/Table file to ensure both fields are filled
//            $user = $this->Users->patchEntity($user, $this->request->getData(), ['validate' => 'resetPassword']);
//            if ($this->Users->save($user)) {
//                $this->Flash->success('The user has been saved.');
//
//                return $this->redirect(['controller' => 'Users', 'action' => 'index']);
//            }
//            $this->Flash->error('The user could not be saved. Please, try again.');
//        }
//        $this->set(compact('user'));
//    }

    /**
     * Login method
     *
     * @return \Cake\Http\Response|null|void Redirect to location before authentication
     */




    public function login()
    {
        $this->request->allowMethod(['get', 'post']);
        $result = $this->Authentication->getResult();

        // Check if the user has failed login attempts and needs to wait
        if ($this->request->is('post')) {
            $email = $this->request->getData('email');
            $session = $this->request->getSession();
            $failedLoginKey = 'failed_login_' . md5($email);
            $lastAttemptKey = 'last_attempt_' . md5($email);

            // Get the current time
            $currentTime = time();

            // Check if there was a recent failed attempt
            $lastAttemptTime = $session->read($lastAttemptKey);
            if ($lastAttemptTime && ($currentTime - $lastAttemptTime) < 3) {
                // User must wait for the full 20 seconds
                $timeToWait = 3 - ($currentTime - $lastAttemptTime);
                $this->Flash->error(__('Too many failed attempts. Please wait {0} seconds before trying again.', $timeToWait));
                return $this->render();
            }
        }

        $recaptchaResponse = $this->request->getData('g-recaptcha-response');
        $secretKey = '6Lc1kgArAAAAAO7aseHeBXsYLkFrAEtEtK0KK46o';

        $recaptchaUrl = 'https://www.google.com/recaptcha/api/siteverify';
        $data = [
            'secret' => $secretKey,
            'response' => $recaptchaResponse,
            'remoteip' => $this->request->clientIp()
        ];

        $http = new \Cake\Http\Client();
        $response = $http->post($recaptchaUrl, $data);
        $recaptchaResult = json_decode($response->getBody()->getContents());

        // If user passes authentication, grant access to the system
        if ($recaptchaResult->success && $result && $result->isValid()) {
            // Clear any failed login records for this user
            if (!empty($email)) {
                $session = $this->request->getSession();
                $failedLoginKey = 'failed_login_' . md5($email);
                $lastAttemptKey = 'last_attempt_' . md5($email);
                $session->delete($failedLoginKey);
                $session->delete($lastAttemptKey);
            }

            // Set a fallback location in case user logged in without triggering 'unauthenticatedRedirect'
            $fallbackLocation = ['controller' => 'Programs', 'action' => 'index'];
            // And redirect user to the location they're trying to access
            return $this->redirect($this->Authentication->getLoginRedirect() ?? $fallbackLocation);
        }

        if (!$recaptchaResult->success && $this->request->is('post')) {
            $this->Flash->error('Verify the Captcha');
        }

        if ($this->request->is('post')) {
            if (!$recaptchaResult->success) {
                $this->Authentication->logout();
                $this->Flash->error(__('reCAPTCHA verification failed. Please try again.'));
                return $this->render();
            }

            // Display error if user submitted their credentials but authentication failed
            if ($recaptchaResult->success && !$result->isValid()) {
                $email = $this->request->getData('email');

                // Record this failed attempt
                $session = $this->request->getSession();
                $failedLoginKey = 'failed_login_' . md5($email);
                $lastAttemptKey = 'last_attempt_' . md5($email);

                // Update the timestamp of the last failed attempt
                $session->write($lastAttemptKey, time());

                // Increment the counter for failed attempts
                $failedCount = $session->read($failedLoginKey) ?? 0;
                $session->write($failedLoginKey, $failedCount + 1);

                $this->Flash->error('Email address and/or Password is incorrect. Please try again.');
            }
        }
    }

//    private function verifyCaptcha(){
//
//        $recaptchaResponse = $this->request->getData('g-recaptcha-response');
//        $secretKey = '6Le-QvsqAAAAAEMV6nJFl2cuj8NR1VWtDM3thckP';
//
//
//        $recaptchaUrl = 'https://www.google.com/recaptcha/api/siteverify';
//        $data = [
//            'secret' => $secretKey,
//            'response' => $recaptchaResponse,
//            'remoteip' => $this->request->clientIp()
//        ];
//
//        $http = new \Cake\Http\Client();
//        $response = $http->post($recaptchaUrl, $data);
//        $recaptchaResult = json_decode($response->getBody()->getContents());
//    }
    /**
     * Logout method
     *
     * @return \Cake\Http\Response|null|void
     */
    public function logout()
    {
        // We only need to log out a user when they're logged in
        $result = $this->Authentication->getResult();
        if ($result && $result->isValid()) {
            $this->Authentication->logout();

            $this->Flash->success('You have been logged out successfully. ');
        }

        // Otherwise just send them to the login page
        return $this->redirect('/');
        }
}
