<?php
namespace App\Controller;

use App\Controller\AppController;

class DonationsController extends AppController
{
    public function index()
    {
        $this->viewBuilder()->setLayout('adminlayout');

        // Fetch the Donations table
        $donationsTable = $this->fetchTable('Donations');

        $search = $this->request->getQuery('search');
        $query = $donationsTable->find()->contain(['Campaigns']);
        if (!empty($search)) {
            $query->where([
                'OR' => [
                    'donor_name LIKE' => '%' . $search . '%',
                    'donor_email LIKE' => '%' . $search . '%',
                    'stripe_transaction_id LIKE' => '%' . $search . '%',
                ]
            ]);
        }

        $this->paginate = [
            'limit' => 1000,
            'order' => ['created' => 'desc']
        ];
        $donations = $this->paginate($query);
        $sumQuery = $donationsTable->find()->select([
            'sum' => $donationsTable->find()->func()->sum('amount')
        ]);
        if (!empty($search)) {
            $sumQuery->where([
                'OR' => [
                    'donor_name LIKE' => '%' . $search . '%',
                    'donor_email LIKE' => '%' . $search . '%',
                    'stripe_transaction_id LIKE' => '%' . $search . '%',
                ]
            ]);
        }
        $totalAmount = $sumQuery->first()->sum ?? 0;
        $this->set(compact('donations', 'totalAmount'));
        $this->render('/donations/index');

    }

    public function delete($id = null)
    {
        // Allow only POST and DELETE methods for deletion
        $this->request->allowMethod(['post', 'delete']);

        // Fetch the Donations table
        $donationsTable = $this->fetchTable('Donations');

        // Retrieve the donation entity by ID
        $donation = $donationsTable->get($id);
        if ($donationsTable->delete($donation)) {
            $this->Flash->success(__('The donation has been deleted.'));
        } else {
            $this->Flash->error(__('The donation could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }
    
    public function bulkDelete()
    {
        $this->request->allowMethod(['post']);

        $ids = $this->request->getData('selected_ids');
        if (!empty($ids)) {
            $donationsTable = $this->fetchTable('Donations');
            $query = $donationsTable->find()->where(['id IN' => $ids]);
            $deleted = 0;
            foreach ($query as $donation) {
                if ($donationsTable->delete($donation)) {
                    $deleted++;
                }
            }
            $this->Flash->success("{$deleted} donation(s) deleted successfully.");
        } else {
            $this->Flash->error('No donations were selected for deletion.');
        }

        return $this->redirect(['action' => 'index']);
    }
    
    public function view($id = null)
    {
        $this->viewBuilder()->setLayout('adminlayout');

        $donationsTable = $this->fetchTable('Donations');

        // Retrieve the donation record with associated Campaign
        $donation = $donationsTable->get($id, [
            'contain' => ['Campaigns'],
        ]);

        $this->set(compact('donation'));
        $this->render('/donations/view');

    }
}