<?php
namespace App\Controller;


use App\Controller\AppController;
use Cake\Mailer\Mailer;
use Cake\ORM\TableRegistry;


class TemplatesController extends AppController
    {
        public function view($id = null)
    {
        $this->viewBuilder()->setLayout('adminlayout');
        
        $template = $this->Templates->get($id);

        // Handle program filter from query param
        $programId = $this->request->getQuery('program_id');

        // Redirect to set default if no program_id is provided
        if ($programId === null || $programId === '') {
            return $this->redirect(['action' => 'view', $id, '?' => ['program_id' => 'all']]);
        }

        $students = $volunteers = $visitors = [];

        $studentsTable = $this->fetchTable('Students');
        $volunteersTable = $this->fetchTable('Volunteers');
        $visitorsTable = $this->fetchTable('Visitors');

        if ($programId !== 'all') {
            // Students linked to the selected program
            $students = $studentsTable->find()
                ->matching('Programs', function ($q) use ($programId) {
                    return $q->where(['Programs.id' => $programId]);
                })
                ->contain(['Programs'])
                ->all();

            // Volunteers linked to the selected program
            $volunteers = $volunteersTable->find()
                ->matching('Programs', function ($q) use ($programId) {
                    return $q->where(['Programs.id' => $programId]);
                })
                ->contain(['Programs'])
                ->all();

            // Visitors are not shown when a specific program is selected
            $visitors = [];
        } else {
            // No program filter, show all contacts
            $students = $studentsTable->find()->contain(['Programs'])->all();
            $volunteers = $volunteersTable->find()->contain(['Programs'])->all();
            $visitors = $visitorsTable->find()->all();
        }

        // Get list of programs for filter dropdown
        $programs = TableRegistry::getTableLocator()->get('Programs')
            ->find()
            ->select(['id', 'name'])
            ->toArray();

        $this->set(compact('template', 'students', 'volunteers', 'visitors', 'programs', 'programId'));
    }

        





        public function index()
        {
            $this->viewBuilder()->setLayout('adminlayout');
            $templates = $this->getTableLocator()->get('Templates')->find('all');

            $this->set(compact('templates'));
        }
        public function delete($id = null)
        {
            $this->request->allowMethod(['post', 'delete']);

            $templatesTable = $this->getTableLocator()->get('Templates');
            $recipientsTable = $this->getTableLocator()->get('TemplateRecipients');

            $template = $templatesTable->get($id);

            $recipientsTable->deleteAll(['template_id' => $id]);

            if ($templatesTable->delete($template)) {
                $this->Flash->success(__('The template has been deleted.'));
            } else {
                $this->Flash->error(__('Unable to delete the template.'));
            }

            return $this->redirect(['action' => 'index']);
        }



        public function create()
        {
            $this->viewBuilder()->setLayout('adminlayout');
            $templatesTable = $this->getTableLocator()->get('Templates');
            $template = $templatesTable->newEmptyEntity();

            if ($this->request->is('post')) {
                $template = $templatesTable->patchEntity($template, $this->request->getData());
                if ($templatesTable->save($template)) {
                    $this->Flash->success(__('Template has been created.'));
                    return $this->redirect(['action' => 'index']);
                }
                $this->Flash->error(__('Unable to create template.'));
            }

            $this->set(compact('template'));
        }

        public function edit($id = null)
        {
            $this->viewBuilder()->setLayout('adminlayout');
            $templatesTable = $this->getTableLocator()->get('Templates');
            $template = $templatesTable->get($id);

            if ($this->request->is(['post', 'put', 'patch'])) {
                $templatesTable->patchEntity($template, $this->request->getData());
                if ($templatesTable->save($template)) {
                    $this->Flash->success(__('Template updated.'));
                    return $this->redirect(['action' => 'index']);
                }
                $this->Flash->error(__('Unable to update template.'));
            }

            $this->set(compact('template'));
        }


        public function send()
{
    if ($this->request->is('post')) {
        $data = $this->request->getData();

        // Get template ID and recipients from the form
        $templateId = $data['template_ids'][0] ?? null;
        
        // Decode the recipients data (expected to be in JSON format)
        $recipientsRaw = $data['recipient_emails'] ?? '[]'; // In case it's empty, default to '[]'
        $decoded = json_decode($recipientsRaw, true);
        $recipients = array_map(fn($item) => trim($item['value']), $decoded ?? []);

        
        
        if (!$templateId || empty($recipients)) {
            $this->Flash->error(__('Missing template or recipients.'));
            return $this->redirect($this->referer());
        }

        // Fetch the template directly from the table
        $templatesTable = $this->fetchTable('Templates');
        try {
            $template = $templatesTable->get($templateId);
        } catch (\Cake\Datasource\Exception\RecordNotFoundException $e) {
            $this->Flash->error("Template not found.");
            return $this->redirect($this->referer());
        }

        $sentCount = 0;
        $failed = [];

        // Loop through each recipient email
        foreach ($recipients as $email) {
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $failed[] = $email;
                continue;
            }

            // Create the mailer instance
            $mailer = new Mailer('default');
            
            // Prepare the email content
            $emailContent = $template->content;

            // Check if the email is in the visitors table
            $visitorsTable = $this->fetchTable('Visitors');
            $visitor = $visitorsTable->findByEmail($email)->first();

            // If the email is found in the visitors table, append the unsubscribe footer
            if ($visitor) {
                $unsubscribeUrl = 'http://review.u24s2118.iedev.org/team118-app_fit3048/unsubscribe?email=' . urlencode($email);
                $emailContent .= "
                    <hr style=\"border: 1px solid #ddd; margin: 20px 0;\">
                    <p style=\"font-size: 14px; color: #555; text-align: center;\">You are receiving this email because you subscribed to our promotional emails. If you no longer wish to receive these emails, you can unsubscribe below:</p>
                    <div style=\"text-align: center; margin-top: 20px;\">
                        <a href=\"{$unsubscribeUrl}\" style=\"background-color: #ff4d4d; color: white; padding: 12px 30px; border-radius: 25px; font-size: 16px; text-decoration: none; display: inline-block; font-weight: bold; transition: background-color 0.3s ease;\">
                            Unsubscribe from Promotional Emails
                        </a>
                    </div>
                ";
            }
            

            try {
                $success = $mailer
                    ->setTo($email)
                    ->setSubject($template->name ?: 'No Subject')
                    ->setEmailFormat('html')
                    ->deliver($emailContent);
            
                if ($success) {
                    $sentCount++;
                } else {
                    $failed[] = $email;
                }
            } catch (\Exception $e) {
                $failed[] = $email;
                \Cake\Log\Log::error('Email error: ' . $e->getMessage());
            }
        }

        // Provide feedback to the user
        $this->Flash->success("Email sent to {$sentCount} recipient(s).");

        if (!empty($failed)) {
            $this->Flash->error("Some emails failed: " . implode(', ', $failed));
        }

        return $this->redirect(['action' => 'index']);
    }

    // Handle invalid request method
    $this->Flash->error('Invalid request method.');
    return $this->redirect($this->referer());
}





        public function uploadImage()
{
    $this->request->allowMethod(['post']);
    $file = $this->request->getData('file');

    if (!$file) {
        throw new \Cake\Http\Exception\BadRequestException('Missing file.');
    }

    $filename = time() . '-' . $file->getClientFilename();
    $targetPath = WWW_ROOT . 'uploads' . DS . $filename;
    $file->moveTo($targetPath);

    $publicUrl = Router::url('/uploads/' . $filename, true);

    $this->viewBuilder()->setLayout('ajax');
    $this->set(compact('publicUrl'));
    $this->set('_serialize', ['location' => $publicUrl]);

    $this->response = $this->response->withType('application/json');
    $this->response = $this->response->withStringBody(json_encode([
        'location' => $publicUrl
    ]));

    return $this->response;
}





    }


