<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Query\SelectQuery;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

/**
 * ProgramsStudents Model
 *
 * @property \App\Model\Table\ProgramsTable&\Cake\ORM\Association\BelongsTo $Programs
 * @property \App\Model\Table\StudentsTable&\Cake\ORM\Association\BelongsTo $Students
 *
 * @method \App\Model\Entity\ProgramsStudent newEmptyEntity()
 * @method \App\Model\Entity\ProgramsStudent newEntity(array $data, array $options = [])
 * @method array<\App\Model\Entity\ProgramsStudent> newEntities(array $data, array $options = [])
 * @method \App\Model\Entity\ProgramsStudent get(mixed $primaryKey, array|string $finder = 'all', \Psr\SimpleCache\CacheInterface|string|null $cache = null, \Closure|string|null $cacheKey = null, mixed ...$args)
 * @method \App\Model\Entity\ProgramsStudent findOrCreate($search, ?callable $callback = null, array $options = [])
 * @method \App\Model\Entity\ProgramsStudent patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method array<\App\Model\Entity\ProgramsStudent> patchEntities(iterable $entities, array $data, array $options = [])
 * @method \App\Model\Entity\ProgramsStudent|false save(\Cake\Datasource\EntityInterface $entity, array $options = [])
 * @method \App\Model\Entity\ProgramsStudent saveOrFail(\Cake\Datasource\EntityInterface $entity, array $options = [])
 * @method iterable<\App\Model\Entity\ProgramsStudent>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\ProgramsStudent>|false saveMany(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\ProgramsStudent>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\ProgramsStudent> saveManyOrFail(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\ProgramsStudent>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\ProgramsStudent>|false deleteMany(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\ProgramsStudent>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\ProgramsStudent> deleteManyOrFail(iterable $entities, array $options = [])
 */
class ProgramsStudentsTable extends Table
{
        /**
     * Initialize method
     *
     * @param array<string, mixed> $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('programs_students');
        $this->setDisplayField('id');
        $this->setPrimaryKey('id');

        $this->belongsTo('Programs', [
            'foreignKey' => 'program_id',
        ]);
        $this->belongsTo('Students', [
            'foreignKey' => 'student_id',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
    {
        $validator
            ->integer('program_id')
            ->notEmptyString('program_id', 'Program ID is required.')
            ->add('program_id', 'validProgram', [
                'rule' => 'numeric',
                'message' => 'Program ID must be a numeric value.'
            ]);

        $validator
            ->integer('student_id')
            ->notEmptyString('student_id', 'Student ID is required.')
            ->add('student_id', 'validStudent', [
                'rule' => 'numeric',
                'message' => 'Student ID must be a numeric value.'
            ]);

          
        $validator
            ->scalar('name')
            ->maxLength('name', 50, 'Status name must not exceed 50 characters.')
            ->requirePresence('name', 'create')
             ->notEmptyString('name', 'Program status name is required.')
            ->add('name', 'alphaOnly', [
                   'rule' => function ($value, $context) {
                        return preg_match('/^[a-zA-Z\s]+$/', $value) === 1; 
                    },
                        'message' => 'Only letters and spaces are allowed in the program status name.'
                ]);
            
                return $validator;
            }
    

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->existsIn(['program_id'], 'Programs'), ['errorField' => 'program_id']);
        $rules->add($rules->existsIn(['student_id'], 'Students'), ['errorField' => 'student_id']);

        return $rules;
    }
}


