<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Query\SelectQuery;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

/**
 * Volunteers Model
 */
class VolunteersTable extends Table
{
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('volunteers');
        $this->setDisplayField('first_name'); // ✅ Fixed: removed stray "7"
        $this->setPrimaryKey('id');

        $this->belongsTo('VolunteerStatuses', [
            'foreignKey' => 'volunteer_status_id',
        ]);

        $this->belongsToMany('Programs', [
            'foreignKey' => 'volunteer_id',
            'targetForeignKey' => 'program_id',
            'joinTable' => 'programs_volunteers',
        ]);
    }

    public function validationDefault(Validator $validator): Validator
    {
        $validator
            ->requirePresence('first_name')
            ->notEmptyString('first_name')
            ->add('first_name', 'validChars', [
                'rule' => ['custom', "/^[a-zA-Z\s'-]+$/"],
                'message' => "Only letters, spaces, apostrophes and dashes are allowed."
            ])

            ->requirePresence('last_name')
            ->notEmptyString('last_name')
            ->add('last_name', 'validChars', [
                'rule' => ['custom', "/^[a-zA-Z\s'-]+$/"],
                'message' => "Only letters, spaces, apostrophes and dashes are allowed."
            ])

            ->requirePresence('email')
            ->notEmptyString('email')
            ->email('email', false, 'Please provide a valid email.')

            ->requirePresence('phone')
            ->notEmptyString('phone')
            ->add('phone', 'length', [
                'rule' => ['custom', '/^\d{10}$/'],
                'message' => 'Phone number must be exactly 10 digits.'
            ])

            ->requirePresence('postcode')
            ->notEmptyString('postcode')
            ->add('postcode', 'validPostcode', [
                'rule' => ['custom', '/^\d{4,}$/'],
                'message' => 'Postcode must be at least 4 digits.'
            ])
            
            ->requirePresence('occupation')
            ->notEmptyString('occupation')
            ->add('occupation', 'validChars', [
                'rule' => ['custom', "/^[a-zA-Z\s'-]+$/"],
                'message' => "Only letters, spaces, apostrophes and dashes are allowed."
            ])

            ->requirePresence('description'); // Optional: remove if not strictly required

        return $validator;
    }

    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->isUnique(['email'], 'This email is already taken, please choose another one.'), ['errorField' => 'email']);
        $rules->add($rules->existsIn(['volunteer_status_id'], 'VolunteerStatuses'), ['errorField' => 'volunteer_status_id']);

        return $rules;
    }
}
